'use strict'

module.exports = validate

const set = require('lodash/set')
const get = require('lodash/get')
const HttpError = require('../../request/http-error')
const deprecate = require('../../deprecate')

function validate (endpointParams, options) {
  // Alias are handled before validation, as validation rules
  // ar set the aliased parameter. The `mapTo` property is the other way
  // around, the final parameter name is the mapTo value, but validation
  // rules are on parameter with the mapTo property
  Object.keys(options).forEach(optionName => {
    if (!endpointParams[optionName] || !endpointParams[optionName].alias) {
      return
    }

    set(options, endpointParams[optionName].alias, options[optionName])
    delete options[optionName]

    // right now all parameters with an alias property also have a deprecated
    // property, but that might change in future, so we wrap it in the if block,
    // but ignore if for coverage
    /* istanbul ignore else */
    if (endpointParams[optionName].deprecated) {
      deprecate(`"${optionName}" parameter has been renamed to "${endpointParams[optionName].alias}"`)
    }
  })

  Object.keys(endpointParams).forEach(parameterName => {
    const parameter = get(endpointParams, parameterName)
    const expectedType = parameter.type
    let parentParameterName
    let parentValue
    let parentParamIsPresent = true
    let parentParameterIsArray = false

    if (/\./.test(parameterName)) {
      parentParameterName = parameterName.replace(/\.[^.]+$/, '')
      parentParameterIsArray = parentParameterName.slice(-2) === '[]'
      if (parentParameterIsArray) {
        parentParameterName = parentParameterName.slice(0, -2)
      }
      parentValue = get(options, parentParameterName)
      parentParamIsPresent = parentParameterName === 'headers' || (typeof parentValue === 'object' && parentValue !== null)
    }

    let values = parentParameterIsArray
      ? (get(options, parentParameterName) || []).map(value => value[parameterName.split(/\./).pop()])
      : [get(options, parameterName)]

    values.forEach((value, i) => {
      const valueIsPresent = typeof value !== 'undefined'
      const valueIsNull = value === null
      const currentParameterName = parentParameterIsArray
        ? parameterName.replace(/\[\]/, `[${i}]`)
        : parameterName

      if (!parameter.required && !valueIsPresent) {
        return
      }

      // if the parent parameter is of type object but allows null
      // then the child parameters can be ignored
      if (!parentParamIsPresent) {
        return
      }

      if (parameter.allowNull && valueIsNull) {
        return
      }

      if (!parameter.allowNull && valueIsNull) {
        throw new HttpError(`'${currentParameterName}' cannot be null`, 400)
      }

      if (parameter.required && !valueIsPresent) {
        throw new HttpError(`Empty value for parameter '${currentParameterName}': ${JSON.stringify(value)}`, 400)
      }

      // parse to integer before checking for enum
      // so that string "1" will match enum with number 1
      if (expectedType === 'integer') {
        const unparsedValue = value
        value = parseInt(value, 10)
        if (isNaN(value)) {
          throw new HttpError(`Invalid value for parameter '${currentParameterName}': ${JSON.stringify(unparsedValue)} is NaN`, 400)
        }
      }

      if (parameter.enum && parameter.enum.indexOf(value) === -1) {
        throw new HttpError(`Invalid value for parameter '${currentParameterName}': ${JSON.stringify(value)}`, 400)
      }

      if (parameter.validation) {
        const regex = new RegExp(parameter.validation)
        if (!regex.test(value)) {
          throw new HttpError(`Invalid value for parameter '${currentParameterName}': ${JSON.stringify(value)}`, 400)
        }
      }

      if (expectedType === 'object' && typeof value === 'string') {
        try {
          value = JSON.parse(value)
        } catch (exception) {
          throw new HttpError(`JSON parse error of value for parameter '${currentParameterName}': ${JSON.stringify(value)}`, 400)
        }
      }

      set(options, parameter.mapTo || currentParameterName, value)
    })
  })

  return options
}
